import React, { useState, useEffect, useRef } from "react";
import {
  Brain,
  Trash2,
  Check,
  Trophy,
  Sparkles,
  Pencil,
  Eraser,
} from "lucide-react";

const AIDrawingGame = () => {
  const canvasRef = useRef(null);
  const [isDrawing, setIsDrawing] = useState(false);
  const [currentPrompt, setCurrentPrompt] = useState("");
  const [score, setScore] = useState(0);
  const [gameActive, setGameActive] = useState(false);
  const [predictions, setPredictions] = useState([]);
  const [streak, setStreak] = useState(0);
  const [bestStreak, setBestStreak] = useState(0);
  const [brushSize, setBrushSize] = useState(8);
  const [toolMode, setToolMode] = useState("draw");
  const strokes = useRef([]);

  const prompts = [
    "circle",
    "square",
    "triangle",
    "star",
    "heart",
    "house",
    "tree",
    "sun",
    "moon",
    "cloud",
    "flower",
    "car",
    "smile",
    "arrow",
    "diamond",
    "spiral",
    "wave",
    "mountain",
    "fish",
    "bird",
    "cat",
    "dog",
    "apple",
    "cup",
  ];

  // Helper functions for pattern recognition
  const getAverageDistanceFromCenter = (strokes) => {
    if (strokes.length === 0) return 0;
    const centerX = strokes.reduce((sum, p) => sum + p.x, 0) / strokes.length;
    const centerY = strokes.reduce((sum, p) => sum + p.y, 0) / strokes.length;
    return (
      strokes.reduce((sum, p) => {
        return (
          sum +
          Math.sqrt(Math.pow(p.x - centerX, 2) + Math.pow(p.y - centerY, 2))
        );
      }, 0) / strokes.length
    );
  };

  const getDistanceVariance = (strokes) => {
    if (strokes.length === 0) return 0;
    const avgDist = getAverageDistanceFromCenter(strokes);
    const centerX = strokes.reduce((sum, p) => sum + p.x, 0) / strokes.length;
    const centerY = strokes.reduce((sum, p) => sum + p.y, 0) / strokes.length;
    const variance =
      strokes.reduce((sum, p) => {
        const dist = Math.sqrt(
          Math.pow(p.x - centerX, 2) + Math.pow(p.y - centerY, 2)
        );
        return sum + Math.pow(dist - avgDist, 2);
      }, 0) / strokes.length;
    return Math.sqrt(variance);
  };

  const detectCorners = (strokes) => {
    if (strokes.length < 10) return 0;
    let corners = 0;
    for (let i = 5; i < strokes.length - 5; i++) {
      const before = {
        x: strokes[i - 5].x - strokes[i].x,
        y: strokes[i - 5].y - strokes[i].y,
      };
      const after = {
        x: strokes[i + 5].x - strokes[i].x,
        y: strokes[i + 5].y - strokes[i].y,
      };
      const angle = Math.abs(
        Math.atan2(before.y, before.x) - Math.atan2(after.y, after.x)
      );
      if (angle > Math.PI / 3 && angle < (2 * Math.PI) / 3) corners++;
    }
    return corners;
  };

  const detectSharpPoints = (strokes) => {
    if (strokes.length < 10) return 0;
    let points = 0;
    for (let i = 3; i < strokes.length - 3; i++) {
      const before = {
        x: strokes[i - 3].x - strokes[i].x,
        y: strokes[i - 3].y - strokes[i].y,
      };
      const after = {
        x: strokes[i + 3].x - strokes[i].x,
        y: strokes[i + 3].y - strokes[i].y,
      };
      const angle = Math.abs(
        Math.atan2(before.y, before.x) - Math.atan2(after.y, after.x)
      );
      if (angle > (2 * Math.PI) / 3) points++;
    }
    return points;
  };

  const detectUpwardCurve = (strokes) => {
    if (strokes.length < 5) return false;
    const sorted = [...strokes].sort((a, b) => a.x - b.x);
    const leftY = sorted[0].y;
    const midY = sorted[Math.floor(sorted.length / 2)].y;
    const rightY = sorted[sorted.length - 1].y;
    return midY > leftY && midY > rightY;
  };

  const detectStraightLine = (strokes) => {
    if (strokes.length < 5) return false;
    const dx = strokes[strokes.length - 1].x - strokes[0].x;
    const dy = strokes[strokes.length - 1].y - strokes[0].y;
    const expectedLength = Math.sqrt(dx * dx + dy * dy);
    let actualLength = 0;
    for (let i = 1; i < strokes.length; i++) {
      const segDx = strokes[i].x - strokes[i - 1].x;
      const segDy = strokes[i].y - strokes[i - 1].y;
      actualLength += Math.sqrt(segDx * segDx + segDy * segDy);
    }
    return actualLength < expectedLength * 1.3;
  };

  const patterns = {
    circle: (strokes) => {
      const avgDist = getAverageDistanceFromCenter(strokes);
      const variance = getDistanceVariance(strokes);
      return variance < avgDist * 0.3 ? 0.9 : 0.4;
    },
    square: (strokes) => {
      const corners = detectCorners(strokes);
      return corners >= 3 ? 0.85 : 0.3;
    },
    triangle: (strokes) => {
      const corners = detectCorners(strokes);
      return corners === 3 ? 0.9 : 0.35;
    },
    star: (strokes) => {
      const points = detectSharpPoints(strokes);
      return points >= 4 ? 0.85 : 0.3;
    },
    heart: (strokes) => {
      if (strokes.length < 10) return 0.3;
      const canvas = canvasRef.current;
      const topThird = strokes.filter((p) => p.y < canvas.height / 3);
      const maxY = Math.max(...strokes.map((p) => p.y));
      const bottomPoints = strokes.filter((p) => p.y > maxY - 20);
      return topThird.length > strokes.length * 0.3 &&
        bottomPoints.length < strokes.length * 0.2
        ? 0.8
        : 0.3;
    },
    house: (strokes) => {
      const canvas = canvasRef.current;
      const topThird = strokes.filter((p) => p.y < canvas.height / 3);
      const bottomHalf = strokes.filter((p) => p.y > canvas.height / 2);
      return topThird.length > 0 && bottomHalf.length > strokes.length * 0.3
        ? 0.85
        : 0.3;
    },
    tree: (strokes) => {
      const canvas = canvasRef.current;
      const topHalf = strokes.filter((p) => p.y < canvas.height / 2);
      const bottomQuarter = strokes.filter((p) => p.y > canvas.height * 0.75);
      return topHalf.length > strokes.length * 0.4 && bottomQuarter.length > 5
        ? 0.75
        : 0.3;
    },
    sun: (strokes) => {
      const avgDist = getAverageDistanceFromCenter(strokes);
      const variance = getDistanceVariance(strokes);
      const points = detectSharpPoints(strokes);
      return variance < avgDist * 0.4 && points > 3 ? 0.8 : 0.3;
    },
    smile: (strokes) => {
      return detectUpwardCurve(strokes) ? 0.85 : 0.3;
    },
    arrow: (strokes) => {
      const hasLine = detectStraightLine(strokes);
      const hasHead = detectSharpPoints(strokes) >= 1;
      return hasLine && hasHead ? 0.85 : 0.3;
    },
    spiral: (strokes) => {
      if (strokes.length < 20) return 0.3;
      const centerX = strokes.reduce((sum, p) => sum + p.x, 0) / strokes.length;
      const centerY = strokes.reduce((sum, p) => sum + p.y, 0) / strokes.length;

      let increasing = 0;
      for (let i = 10; i < strokes.length; i++) {
        const dist1 = Math.sqrt(
          Math.pow(strokes[i - 10].x - centerX, 2) +
            Math.pow(strokes[i - 10].y - centerY, 2)
        );
        const dist2 = Math.sqrt(
          Math.pow(strokes[i].x - centerX, 2) +
            Math.pow(strokes[i].y - centerY, 2)
        );
        if (dist2 > dist1) increasing++;
      }
      return increasing > strokes.length * 0.4 ? 0.8 : 0.3;
    },
    wave: (strokes) => {
      if (strokes.length < 15) return 0.3;
      const sorted = [...strokes].sort((a, b) => a.x - b.x);
      let peaks = 0;
      let valleys = 0;

      for (let i = 3; i < sorted.length - 3; i++) {
        const before = sorted[i - 3].y;
        const current = sorted[i].y;
        const after = sorted[i + 3].y;

        if (current < before && current < after) valleys++;
        if (current > before && current > after) peaks++;
      }

      return peaks >= 2 && valleys >= 2 ? 0.85 : 0.3;
    },
    diamond: (strokes) => {
      const corners = detectCorners(strokes);
      return corners === 4 ? 0.85 : 0.3;
    },
    mountain: (strokes) => {
      if (strokes.length < 10) return 0.3;
      const sorted = [...strokes].sort((a, b) => a.x - b.x);
      const peaks = detectSharpPoints(sorted);
      return peaks >= 2 ? 0.75 : 0.3;
    },
  };

  const recognizeDrawing = (strokeData) => {
    if (strokeData.length < 5) {
      setPredictions([]);
      return;
    }

    const results = [];
    for (const name of Object.keys(patterns)) {
      const recognizer = patterns[name];
      const confidence = recognizer(strokeData);
      results.push({ name, confidence });
    }

    results.sort((a, b) => b.confidence - a.confidence);
    setPredictions(results.slice(0, 5));

    if (
      gameActive &&
      results[0].name === currentPrompt &&
      results[0].confidence > 0.7
    ) {
      handleCorrectDrawing();
    }
  };

  const handleCorrectDrawing = () => {
    setScore((prev) => prev + 10 + streak * 2);
    setStreak((prev) => prev + 1);
    setBestStreak((prev) => Math.max(prev, streak + 1));
    clearCanvas();
    setCurrentPrompt(prompts[Math.floor(Math.random() * prompts.length)]);
  };

  const startGame = () => {
    setGameActive(true);
    setScore(0);
    setStreak(0);
    setCurrentPrompt(prompts[Math.floor(Math.random() * prompts.length)]);
    clearCanvas();
    setPredictions([]);
  };

  const endGame = () => {
    setGameActive(false);
  };

  const clearCanvas = () => {
    const canvas = canvasRef.current;
    if (!canvas) return;
    const ctx = canvas.getContext("2d");
    ctx.fillStyle = "#ffffff";
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    strokes.current = [];
    setPredictions([]);
  };

  useEffect(() => {
    const canvas = canvasRef.current;
    const ctx = canvas.getContext("2d");

    const updateSize = () => {
      const container = canvas.parentElement;
      canvas.width = container.offsetWidth;
      canvas.height = container.offsetHeight;
      clearCanvas();
    };

    updateSize();
    window.addEventListener("resize", updateSize);

    return () => {
      window.removeEventListener("resize", updateSize);
    };
  }, []);

  const startDrawing = (e) => {
    // Only allow drawing if game is active
    if (!gameActive) return;

    setIsDrawing(true);
    const rect = canvasRef.current.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;
    strokes.current = [{ x, y }];
  };

  const draw = (e) => {
    // Only allow drawing if game is active
    if (!isDrawing || !gameActive) return;

    const canvas = canvasRef.current;
    const ctx = canvas.getContext("2d");
    const rect = canvas.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;

    if (toolMode === "draw") {
      ctx.strokeStyle = "#1f2937";
      ctx.lineWidth = brushSize;
      ctx.lineCap = "round";
      ctx.lineJoin = "round";

      ctx.beginPath();
      ctx.moveTo(
        strokes.current[strokes.current.length - 1].x,
        strokes.current[strokes.current.length - 1].y
      );
      ctx.lineTo(x, y);
      ctx.stroke();

      strokes.current.push({ x, y });
    } else {
      ctx.fillStyle = "#ffffff";
      ctx.beginPath();
      ctx.arc(x, y, brushSize * 2, 0, Math.PI * 2);
      ctx.fill();
    }
  };

  const stopDrawing = () => {
    if (isDrawing && toolMode === "draw" && gameActive) {
      recognizeDrawing(strokes.current);
    }
    setIsDrawing(false);
  };

  return (
    <div className="w-full h-screen bg-gradient-to-br from-purple-900 via-blue-900 to-indigo-900 flex flex-col">
      <div className="bg-black/30 backdrop-blur-sm border-b border-white/10 p-3 sm:p-4">
        <div className="max-w-7xl mx-auto flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3">
          <div>
            <h1 className="text-xl sm:text-2xl lg:text-3xl font-bold text-white flex items-center gap-2">
              <Brain className="w-6 h-6 sm:w-8 sm:h-8 text-purple-400" />
              AI Drawing Recognition
            </h1>
            {gameActive && (
              <p className="text-purple-300 text-sm mt-1">
                Draw:{" "}
                <span className="font-bold text-white capitalize">
                  {currentPrompt}
                </span>
              </p>
            )}
          </div>

          <div className="flex items-center gap-4 flex-wrap">
            {gameActive && (
              <>
                <div className="flex items-center gap-2 bg-white/10 px-3 py-2 rounded-lg">
                  <Trophy className="w-4 h-4 text-yellow-400" />
                  <span className="text-white font-bold">{score}</span>
                </div>
                <div className="flex items-center gap-2 bg-white/10 px-3 py-2 rounded-lg">
                  <Sparkles className="w-4 h-4 text-orange-400" />
                  <span className="text-white font-bold">Streak: {streak}</span>
                </div>
                <button
                  onClick={endGame}
                  className="px-6 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg font-semibold transition-all transform hover:scale-105"
                >
                  End Game
                </button>
              </>
            )}
            {!gameActive && (
              <button
                onClick={startGame}
                className="px-6 py-2 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg font-semibold transition-all transform hover:scale-105"
              >
                Start Game
              </button>
            )}
          </div>
        </div>
      </div>

      <div className="flex-1 flex flex-col lg:flex-row overflow-hidden">
        <div className="bg-black/30 backdrop-blur-sm border-b lg:border-b-0 lg:border-r border-white/10 p-4 lg:w-72 overflow-y-auto">
          <div className="space-y-4">
            {!gameActive && (
              <div className="bg-purple-600/20 border border-purple-400/50 rounded-lg p-4 text-center">
                <p className="text-white text-sm">
                  Click "Start Game" to begin drawing!
                </p>
              </div>
            )}

            <div>
              <h3 className="text-white font-semibold mb-3">Tools</h3>
              <div className="flex gap-2">
                <button
                  onClick={() => setToolMode("draw")}
                  disabled={!gameActive}
                  className={`flex-1 p-3 rounded-lg flex items-center justify-center gap-2 transition-all ${
                    toolMode === "draw"
                      ? "bg-purple-600 text-white"
                      : "bg-white/10 text-white/70 hover:bg-white/20"
                  } ${!gameActive ? "opacity-50 cursor-not-allowed" : ""}`}
                >
                  <Pencil className="w-4 h-4" />
                  Draw
                </button>
                <button
                  onClick={() => setToolMode("erase")}
                  disabled={!gameActive}
                  className={`flex-1 p-3 rounded-lg flex items-center justify-center gap-2 transition-all ${
                    toolMode === "erase"
                      ? "bg-purple-600 text-white"
                      : "bg-white/10 text-white/70 hover:bg-white/20"
                  } ${!gameActive ? "opacity-50 cursor-not-allowed" : ""}`}
                >
                  <Eraser className="w-4 h-4" />
                  Erase
                </button>
              </div>
            </div>

            <div>
              <label className="text-white text-sm mb-2 block">
                Brush Size: {brushSize}px
              </label>
              <input
                type="range"
                min="2"
                max="20"
                value={brushSize}
                onChange={(e) => setBrushSize(parseInt(e.target.value))}
                disabled={!gameActive}
                className={`w-full ${
                  !gameActive ? "opacity-50 cursor-not-allowed" : ""
                }`}
              />
            </div>

            <button
              onClick={clearCanvas}
              disabled={!gameActive}
              className={`w-full p-3 bg-red-600 hover:bg-red-700 text-white rounded-lg flex items-center justify-center gap-2 transition-colors ${
                !gameActive ? "opacity-50 cursor-not-allowed" : ""
              }`}
            >
              <Trash2 className="w-4 h-4" />
              Clear Canvas
            </button>

            <div>
              <h3 className="text-white font-semibold mb-3 flex items-center gap-2">
                <Brain className="w-5 h-5 text-purple-400" />
                AI Predictions
              </h3>
              <div className="space-y-2">
                {predictions.length === 0 ? (
                  <p className="text-white/50 text-sm text-center py-4">
                    {gameActive
                      ? "Start drawing..."
                      : "Start game to see predictions"}
                  </p>
                ) : (
                  predictions.map((pred, i) => (
                    <div
                      key={i}
                      className={`p-3 rounded-lg ${
                        i === 0
                          ? "bg-purple-600/30 border border-purple-400/50"
                          : "bg-white/5"
                      }`}
                    >
                      <div className="flex items-center justify-between mb-1">
                        <span className="text-white font-medium capitalize">
                          {pred.name}
                        </span>
                        {gameActive &&
                          pred.name === currentPrompt &&
                          pred.confidence > 0.7 && (
                            <Check className="w-4 h-4 text-green-400" />
                          )}
                      </div>
                      <div className="w-full bg-white/10 rounded-full h-2">
                        <div
                          className={`h-2 rounded-full transition-all ${
                            pred.confidence > 0.7
                              ? "bg-green-400"
                              : "bg-purple-400"
                          }`}
                          style={{ width: `${pred.confidence * 100}%` }}
                        />
                      </div>
                      <span className="text-white/70 text-xs">
                        {(pred.confidence * 100).toFixed(0)}%
                      </span>
                    </div>
                  ))
                )}
              </div>
            </div>

            {bestStreak > 0 && (
              <div className="bg-white/5 rounded-lg p-4">
                <h3 className="text-white font-semibold mb-2">Stats</h3>
                <div className="text-purple-300 text-sm">
                  Best Streak:{" "}
                  <span className="text-white font-bold">{bestStreak}</span>
                </div>
              </div>
            )}
          </div>
        </div>

        <div className="flex-1 p-4 sm:p-6 lg:p-8 flex items-center justify-center">
          <div className="w-full max-w-4xl aspect-[4/3] bg-white rounded-2xl shadow-2xl overflow-hidden relative">
            {!gameActive && (
              <div className="absolute inset-0 bg-gray-100 flex items-center justify-center z-10">
                <div className="text-center">
                  <Brain className="w-16 h-16 text-purple-400 mx-auto mb-4" />
                  <p className="text-gray-600 text-lg font-semibold">
                    Click "Start Game" to begin!
                  </p>
                </div>
              </div>
            )}
            <canvas
              ref={canvasRef}
              onMouseDown={startDrawing}
              onMouseMove={draw}
              onMouseUp={stopDrawing}
              onMouseLeave={stopDrawing}
              onTouchStart={(e) => {
                if (!gameActive) return;
                e.preventDefault();
                const touch = e.touches[0];
                const mouseEvent = new MouseEvent("mousedown", {
                  clientX: touch.clientX,
                  clientY: touch.clientY,
                });
                canvasRef.current.dispatchEvent(mouseEvent);
              }}
              onTouchMove={(e) => {
                if (!gameActive) return;
                e.preventDefault();
                const touch = e.touches[0];
                const mouseEvent = new MouseEvent("mousemove", {
                  clientX: touch.clientX,
                  clientY: touch.clientY,
                });
                canvasRef.current.dispatchEvent(mouseEvent);
              }}
              onTouchEnd={(e) => {
                if (!gameActive) return;
                e.preventDefault();
                const mouseEvent = new MouseEvent("mouseup", {});
                canvasRef.current.dispatchEvent(mouseEvent);
              }}
              className={`w-full h-full ${
                gameActive ? "cursor-crosshair" : "cursor-not-allowed"
              } touch-none`}
            />
          </div>
        </div>
      </div>
    </div>
  );
};

export default AIDrawingGame;
